/* Minimal SPA for EinsatzBoard */

const state = {
  user: null,
  csrf: null,
  settings: null,
  incidents: [],
  incidentId: null,
  sse: null,
  lastEventId: localStorage.getItem('lastEventId') || '0',
  page: 'dashboard',
  chatTargets: {}, // incidentId -> {orgs, users}
  chatPins: {},    // incidentId -> pinned messages
  chatLastSeen: JSON.parse(localStorage.getItem('chatLastSeen') || '{}'),
  chatUnread: JSON.parse(localStorage.getItem('chatUnread') || '{}'),
};

const el = (id) => document.getElementById(id);

// --- Chat cache helpers ---------------------------------------------------
state.chatMsgCache = state.chatMsgCache || {};   // incidentId -> { byId: Map, order: [id], lastCreatedAt: string }

function chatCacheEnsure(incidentId){
  if (!state.chatMsgCache[incidentId]) {
    state.chatMsgCache[incidentId] = { byId: new Map(), order: [], lastCreatedAt: '' };
  }
  return state.chatMsgCache[incidentId];
}

function chatCacheMerge(incidentId, messages){
  const cache = chatCacheEnsure(incidentId);
  if (!Array.isArray(messages)) return cache;
  for (const m of messages) {
    if (!m || !m.id) continue;
    if (!cache.byId.has(m.id)) {
      cache.byId.set(m.id, m);
      cache.order.push(m.id);
    } else {
      // update existing (in case scope/author fields got enriched)
      const prev = cache.byId.get(m.id);
      cache.byId.set(m.id, Object.assign({}, prev, m));
    }
    if (m.created_at && (!cache.lastCreatedAt || String(m.created_at) > String(cache.lastCreatedAt))) {
      cache.lastCreatedAt = String(m.created_at);
    }
  }
  // stable chronological order by created_at then id
  cache.order.sort((a,b)=>{
    const A = cache.byId.get(a);
    const B = cache.byId.get(b);
    const ta = String(A?.created_at||'');
    const tb = String(B?.created_at||'');
    if (ta < tb) return -1;
    if (ta > tb) return 1;
    return String(a).localeCompare(String(b));
  });
  return cache;
}

function chatCacheList(incidentId){
  const cache = chatCacheEnsure(incidentId);
  return cache.order.map(id=>cache.byId.get(id)).filter(Boolean);
}

function isNearBottom(node, px=60){
  if (!node) return true;
  return (node.scrollHeight - node.scrollTop - node.clientHeight) < px;
}

function chatMentionHighlightHtml(text){
  const t = escapeHtml(String(text||''));
  // lightweight highlight for @mentions
  const withMentions = t
    .replace(/(@all|@gfs)/g, '<span class="mention">$1</span>')
    .replace(/(@org:[A-Za-z0-9_-]+)/g, '<span class="mention">$1</span>')
    .replace(/(@user:[A-Za-z0-9._-]+)/g, '<span class="mention">$1</span>');
  // auto-link URLs (keeps it simple)
  return withMentions.replace(/(https?:\/\/[^\s<]+[^<.,;:\"')\]\s])/g, '<a href="$1" target="_blank" rel="noopener">$1</a>');
}

function fmtTime(iso){
  if (!iso) return '';
  const d = new Date(iso);
  if (Number.isNaN(d.getTime())) return String(iso);
  return d.toLocaleTimeString([], {hour:'2-digit', minute:'2-digit'});
}

function fmtDateLabel(iso){
  if (!iso) return '';
  const d = new Date(iso);
  if (Number.isNaN(d.getTime())) return '';
  return d.toLocaleDateString([], {weekday:'short', day:'2-digit', month:'2-digit', year:'numeric'});
}

function scopeBadge(scope, targets, m){
  const s = String(scope||'public');
  if (s==='public') return '<span class="badge scope public">Öffentlich</span>';
  if (s==='all') return '<span class="badge scope all">Alle</span>';
  if (s==='gfs') return '<span class="badge scope gfs">GFs/EL</span>';
  if (s==='org') {
    const to = m?.to_org_id;
    let lbl = 'Org';
    if (to && targets?.orgs) {
      const o = targets.orgs.find(x=>String(x.id)===String(to));
      if (o) lbl = o.short_name ? `Org: ${o.short_name}` : `Org: ${o.name}`;
    }
    return `<span class="badge scope org">${escapeHtml(lbl)}</span>`;
  }
  if (s==='private') {
    const toU = m?.to_user_id;
    let lbl = 'Privat';
    if (toU && targets?.users) {
      const u = targets.users.find(x=>String(x.id)===String(toU));
      if (u) lbl = `Privat: ${u.display_name||u.username}`;
    }
    return `<span class="badge scope private">${escapeHtml(lbl)}</span>`;
  }
  return `<span class="badge scope">${escapeHtml(s)}</span>`;
}

function chatMsgHitsMe(m){
  const msg = String(m?.message||'');
  if (msg.includes('@all') || msg.includes('@gfs')) return true;
  const myUser = state.user?.username || '';
  if (myUser && msg.includes(`@user:${myUser}`)) return true;
  // org short for @org:...
  const incidentId = state.incidentId;
  const tcache = state.chatTargets?.[incidentId];
  let myOrgShort = '';
  if (tcache && state.user?.org_id) {
    const oo = (tcache.orgs || []).find(o => o.id === state.user.org_id);
    if (oo && oo.short_name) myOrgShort = String(oo.short_name).toLowerCase();
  }
  if (myOrgShort && msg.toLowerCase().includes(`@org:${myOrgShort}`)) return true;
  return false;
}

function chatRenderList(incidentId, listEl){
  if (!listEl) return;
  const stick = isNearBottom(listEl, 80);
  state.chatUi = state.chatUi || {};
  const ui = state.chatUi[incidentId] || { onlyMentions:false, search:'' };
  const q = String(ui.search||'').toLowerCase().trim();
  let msgs = chatCacheList(incidentId);
  if (ui.onlyMentions) msgs = msgs.filter(m => chatMsgHitsMe(m));
  if (q) msgs = msgs.filter(m => String(m.message||'').toLowerCase().includes(q));
  const targets = state.chatTargets?.[incidentId];
  listEl.innerHTML = '';
  let lastDate = '';
  let lastAuthor = '';
  for (const m of msgs) {
    const dateLabel = fmtDateLabel(m.created_at);
    if (dateLabel && dateLabel !== lastDate) {
      const sep = document.createElement('div');
      sep.className = 'chat-day';
      sep.textContent = dateLabel;
      listEl.appendChild(sep);
      lastDate = dateLabel;
      lastAuthor = '';
    }

    const mine = String(m.author_user_id) === String(state.user?.id);
    const compact = (!mine && String(m.author_user_id)===String(lastAuthor));
    lastAuthor = String(m.author_user_id||'');

    const row = document.createElement('div');
    row.className = 'chat-msg' + (mine ? ' mine' : '') + (chatMsgHitsMe(m) ? ' mention-hit' : '') + (compact ? ' compact' : '');
    row.dataset.msgid = m.id;

    const metaLeft = compact ? '' : `<div class="chat-meta">
        <span class="chat-author">${escapeHtml(m.author_name || m.author_username || 'Unbekannt')}</span>
        <span class="chat-org">${escapeHtml(m.author_org||'')}</span>
      </div>`;

    const badge = scopeBadge(m.scope, targets, m);

    const bodyHtml = String(m.message||'').includes('\n')
      ? chatMentionHighlightHtml(String(m.message)).replace(/\n/g,'<br>')
      : chatMentionHighlightHtml(m.message);

    const isPinned = !!(state.chatPins?.[incidentId] || []).find(p => String(p.message_id||p.id) === String(m.id));
    row.innerHTML = `
      <div class="chat-bubble">
        ${metaLeft}
        <div class="chat-body">${bodyHtml}</div>
        <div class="chat-foot">
          ${badge}
          <span class="chat-time">${escapeHtml(fmtTime(m.created_at))}</span>
          <button class="pinbtn ${isPinned?'pinned':''}" title="${isPinned?'Unpin':'Pin'}" data-act="pin" data-id="${escapeHtml(m.id)}">📌</button>
        </div>
      </div>
    `;
    listEl.appendChild(row);
  }

  // wire pin buttons
  listEl.querySelectorAll('.pinbtn').forEach(btn=>{
    btn.onclick = async (e)=>{
      const mid = btn.getAttribute('data-id');
      if (!mid) return;
      const want = !btn.classList.contains('pinned');
      await api(`/api/incidents/${incidentId}/chat/${mid}/pin`, {method:'POST', body:{pinned: want}});
      await chatLoadPins(incidentId);
      // rerender list to reflect pinned state
      chatRenderList(incidentId, listEl);
      chatRenderPins(incidentId);
    };
  });

  if (stick) listEl.scrollTop = listEl.scrollHeight;
}

async function chatPullAndUpdate(incidentId, listEl, {limit=200} = {}){
  const cache = chatCacheEnsure(incidentId);
  const since = cache.lastCreatedAt || '';
  const url = since ? `/api/incidents/${incidentId}/chat?since=${encodeURIComponent(since)}` : `/api/incidents/${incidentId}/chat?limit=${limit}`;
  const res = await api(url);
  chatCacheMerge(incidentId, res.messages || []);
  chatRenderList(incidentId, listEl);
}

async function chatLoadPins(incidentId){
  try {
    const res = await api(`/api/incidents/${incidentId}/chat/pins`);
    state.chatPins = state.chatPins || {};
    state.chatPins[incidentId] = res.pins || [];
  } catch(e){
    // ignore if endpoint not available yet
    state.chatPins = state.chatPins || {};
    state.chatPins[incidentId] = state.chatPins[incidentId] || [];
  }
}

function chatRenderPins(incidentId){
  const wrap = el('chatPins');
  if (!wrap) return;
  const pins = state.chatPins?.[incidentId] || [];
  if (!pins.length) {
    wrap.innerHTML = '';
    wrap.style.display = 'none';
    return;
  }
  wrap.style.display = '';
  wrap.innerHTML = `
    <div class="chat-pins-head">
      <div class="chat-pins-title">📌 Pinned (${pins.length})</div>
    </div>
    <div class="chat-pins-list"></div>
  `;
  const list = wrap.querySelector('.chat-pins-list');
  for (const p of pins) {
    const it = document.createElement('div');
    it.className = 'chat-pin-item';
    const preview = String(p.message||'').slice(0, 140);
    it.innerHTML = `
      <div class="chat-pin-text">${chatMentionHighlightHtml(escapeHtml(preview))}</div>
      <div class="chat-pin-meta">
        <span>${escapeHtml(p.author_name || p.author_username || '')}</span>
        <span>${escapeHtml(fmtTime(p.created_at))}</span>
        <button class="pinbtn pinned" title="Unpin" data-id="${escapeHtml(p.message_id||p.id)}">📌</button>
      </div>
    `;
    it.onclick = (e)=>{
      // don't hijack unpin button click
      if (e.target && e.target.closest('.pinbtn')) return;
      const mid = String(p.message_id||p.id);
      const row = document.querySelector(`.chat-msg[data-msgid="${CSS.escape(mid)}"]`);
      if (row) row.scrollIntoView({block:'center'});
    };
    const btn = it.querySelector('.pinbtn');
    if (btn) {
      btn.onclick = async (e)=>{
        e.stopPropagation();
        const mid = btn.getAttribute('data-id');
        await api(`/api/incidents/${incidentId}/chat/${mid}/pin`, {method:'POST', body:{pinned:false}});
        await chatLoadPins(incidentId);
        chatRenderPins(incidentId);
        // update pinned state in message list
        if (state.page==='chat') chatRenderList(incidentId, el('chatList'));
      };
    }
    list.appendChild(it);
  }
}
const toasts = el('toasts');

function getChatLastSeen(incidentId){
  return state.chatLastSeen?.[incidentId] || '';
}
function setChatLastSeen(incidentId, iso){
  state.chatLastSeen = state.chatLastSeen || {};
  state.chatLastSeen[incidentId] = iso;
  localStorage.setItem('chatLastSeen', JSON.stringify(state.chatLastSeen));
}
function getChatUnread(incidentId){
  if (!incidentId) {
    return Object.values(state.chatUnread||{}).reduce((a,b)=>a+(Number(b)||0),0);
  }
  return Number(state.chatUnread?.[incidentId] || 0);
}
function incChatUnread(incidentId, by){
  if (!incidentId) return;
  state.chatUnread = state.chatUnread || {};
  state.chatUnread[incidentId] = Number(state.chatUnread[incidentId] || 0) + (Number(by)||1);
  localStorage.setItem('chatUnread', JSON.stringify(state.chatUnread));
}
function clearChatUnread(incidentId){
  if (!incidentId) return;
  state.chatUnread = state.chatUnread || {};
  state.chatUnread[incidentId] = 0;
  localStorage.setItem('chatUnread', JSON.stringify(state.chatUnread));
}

function parseMentions(text){
  const out = [];
  const re = /@([a-zA-Z]+)(?::([\w\-]+))?/g;
  let m;
  while ((m = re.exec(text)) !== null) {
    const kind = (m[1]||'').toLowerCase();
    const arg = m[2]||'';
    if (kind==='all') out.push({type:'all'});
    else if (kind==='gfs') out.push({type:'gfs'});
    else if (kind==='org' && arg) out.push({type:'org', key: arg});
    else if (kind==='user' && arg) out.push({type:'user', key: arg});
  }
  return out;
}

// Very small modal helper (no dependencies)
function modal({title, html, okText='OK', cancelText='Abbrechen', onOk}){
  const wrap = document.createElement('div');
  wrap.style.position='fixed';
  wrap.style.inset='0';
  wrap.style.background='rgba(0,0,0,.45)';
  wrap.style.display='flex';
  wrap.style.alignItems='center';
  wrap.style.justifyContent='center';
  wrap.style.zIndex='9999';
  wrap.innerHTML = `
    <div class="card" style="width:min(920px,96vw);max-height:88vh;overflow:auto">
      <h3>${escapeHtml(title||'')}</h3>
      <div class="body">${html||''}
        <hr />
        <div class="row" style="justify-content:flex-end">
          <button class="btn secondary" data-a="cancel">${escapeHtml(cancelText)}</button>
          <button class="btn" data-a="ok">${escapeHtml(okText)}</button>
        </div>
      </div>
    </div>`;
  document.body.appendChild(wrap);
  const close = ()=>wrap.remove();
  wrap.addEventListener('click', (e)=>{ if (e.target===wrap) close(); });
  wrap.querySelector('[data-a="cancel"]').onclick = close;
  wrap.querySelector('[data-a="ok"]').onclick = async ()=>{
    try{ await onOk?.(); } catch(e){ toast('Fehler', e.message); return; }
    close();
  };
}
async function fetchAllActiveOrgsForAdmin(){
  // Admin/EL may fetch all orgs; GF might be forbidden. We degrade gracefully.
  try{
    const r = await api('/api/admin/orgs');
    return (r.orgs||[]).filter(o=>o.is_active!==false);
  } catch(e){
    return null;
  }
}

function getDefaultOrgSelection(orgs){
  // default: user's org checked; if not found, first org.
  const ids = new Set();
  if (state.user?.org_id) ids.add(state.user.org_id);
  if (ids.size===0 && orgs?.[0]?.id) ids.add(orgs[0].id);
  return ids;
}

async function openNewIncidentModal(){
  const titleId = 'ni_title';
  const locId = 'ni_loc';
  const orgWrapId = 'ni_orgs';
  const modeInfo = (state.user?.role==='admin' || state.user?.role==='el') ? '' : '<div class="small">Hinweis: Nur Admin/EL können mehrere Organisationen auswählen.</div>';

  let orgs = await fetchAllActiveOrgsForAdmin();
  if (!orgs){
    // fallback to single org
    orgs = [{id: state.user?.org_id, name: state.user?.org_name || 'Eigene Organisation'}].filter(o=>o.id);
  }
  const defaults = getDefaultOrgSelection(orgs);

  const orgHtml = orgs.map(o=>{
    const checked = defaults.has(o.id) ? 'checked' : '';
    return `<label class="chk"><input type="checkbox" class="ni_org" value="${escapeHtml(o.id)}" ${checked}> <span>${escapeHtml(o.name||o.id)}</span></label>`;
  }).join('');

  modal({
    title:'Neuer Einsatz',
    okText:'Erstellen',
    html: `
      <div class="field"><label>Titel</label><input id="${titleId}" placeholder="z.B. Personensuche, Flächensuche..." /></div>
      <div class="field"><label>Ort</label><input id="${locId}" placeholder="optional" /></div>
      <div class="field"><label>Beteiligte Organisationen</label>
        ${modeInfo}
        <div id="${orgWrapId}" class="org-checklist">${orgHtml}</div>
        <div class="small">Mindestens 1 Organisation muss ausgewählt sein.</div>
      </div>
    `,
    onOk: async ()=>{
      const title = (document.getElementById(titleId)?.value||'').trim();
      if (!title){ toast('Fehler','Titel fehlt'); return false; }
      const loc = (document.getElementById(locId)?.value||'').trim();
      const orgIds = Array.from(document.querySelectorAll('#'+orgWrapId+' .ni_org:checked')).map(x=>x.value);
      if (!orgIds.length){ toast('Fehler','Bitte mindestens 1 Organisation auswählen'); return false; }
      await api('/api/incidents', {method:'POST', body:{title, location_text: loc, org_ids: orgIds}});
      await loadIncidents();
      toast('OK','Einsatz erstellt');
      return true;
    }
  });
}


function toast(title, body, actions=[]) {
  const d = document.createElement('div');
  d.className='toast';
  d.innerHTML = `<div class="t">${escapeHtml(title)}</div><div class="small">${escapeHtml(body||'')}</div>`;
  if (actions.length) {
    const a = document.createElement('div');
    a.className='a';
    for (const act of actions) {
      const b = document.createElement('button');
      b.className = 'btn secondary';
      b.textContent = act.label;
      b.onclick = () => { try{act.onClick();}finally{d.remove();} };
      a.appendChild(b);
    }
    d.appendChild(a);
  }
  toasts.appendChild(d);
  setTimeout(()=>d.remove(), 12000);
}

function escapeHtml(s){
  return String(s).replace(/[&<>"']/g, c => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[c]));
}

async function api(path, {method='GET', body, isForm=false}={}) {
  const headers = {};
  if (!isForm) headers['Content-Type']='application/json';
  if (state.csrf) headers['X-CSRF-Token']=state.csrf;
  const res = await fetch(path, {
    method,
    headers,
    body: body ? (isForm ? body : JSON.stringify(body)) : undefined,
    credentials: 'same-origin',
  });
  const ct = res.headers.get('content-type')||'';
  const data = ct.includes('application/json') ? await res.json() : {ok:false, error: await res.text()};
  if (!res.ok || data.ok===false) {
    throw new Error(data.error || `HTTP ${res.status}`);
  }
  return data;
}

function applyBrand(settings){
  const b = settings?.branding;
  if (!b) return;
  document.documentElement.style.setProperty('--primary', b.primaryColor || '#d6002a');
  document.documentElement.style.setProperty('--accent', b.accentColor || '#111827');
  el('brandName').textContent = b.appName || 'EinsatzBoard';
  document.title = b.appName || 'EinsatzBoard';
  const logo = el('brandLogo');
  if (b.logoPath) { logo.src=b.logoPath; logo.style.display='block'; } else { logo.style.display='none'; }
  if (b.faviconPath) {
    let link = document.querySelector('link[rel="icon"]');
    if (!link) { link=document.createElement('link'); link.rel='icon'; document.head.appendChild(link); }
    link.href=b.faviconPath;
  }
}

function setLoggedInUI(on){
  el('viewLogin').style.display = on ? 'none' : 'block';
  el('viewApp').style.display = on ? 'block' : 'none';
  el('btnLogout').style.display = on ? 'inline-block' : 'none';
}

async function boot(){
  try{
    const me = await api('/api/me');
    state.user = me.user;
    state.csrf = me.csrf;
    state.settings = me.settings;
    applyBrand(state.settings);

    if (!state.user) {
      setLoggedInUI(false);
      const pub = await api('/api/public-settings');
      applyBrand(pub.settings);
      el('loginHint').textContent = 'Default seed: admin / admin123 (bitte danach ändern)';
      return;
    }

    el('brandOrg').textContent = `${state.user.display_name} · ${state.user.role}`;
    setLoggedInUI(true);
    await loadIncidents();
    buildNav();
    connectSSE();
    render();
  } catch(e){
    console.error(e);
    toast('Fehler', e.message);
    setLoggedInUI(false);
  }
}

async function loadIncidents(){
  const res = await api('/api/incidents');
  state.incidents = res.incidents;
  const sel = el('incidentSelect');
  sel.innerHTML='';
  const opt0 = document.createElement('option');
  opt0.value='';
  opt0.textContent = '— Einsatz wählen —';
  sel.appendChild(opt0);
  for (const i of state.incidents) {
    const o = document.createElement('option');
    o.value=i.id;
    o.textContent = `${i.title} · ${i.status}`;
    sel.appendChild(o);
  }
  const saved = localStorage.getItem('incidentId');
  if (saved && state.incidents.some(i=>i.id===saved)) state.incidentId=saved;
  sel.value = state.incidentId || '';
}

function buildNav(){
  const pages = [
    ['dashboard','Dashboard'],
    ['participants','Teilnehmer'],
    ['teams','Teams'],
    ['orders','Aufträge'],
    ['inbox','Inbox'],
    ['chat','Chat'],
  ];
  if (state.user?.role==='admin') pages.push(['admin','Admin']);

  const nav = el('nav');
  nav.innerHTML='';
  for (const [key,label] of pages) {
    const b = document.createElement('button');
    let txt = label;
    if (key === 'chat') {
      const c = getChatUnread(state.incidentId);
      if (c > 0) txt = `${label} (${c})`;
    }
    b.textContent = txt;
    b.className = state.page===key ? 'active' : '';
    b.onclick = ()=>{ state.page=key; buildNav(); render(); };
    nav.appendChild(b);
  }
}

function requireIncident(){
  if (!state.incidentId) throw new Error('Bitte zuerst einen Einsatz auswählen');
}

async function render(){
  const titleMap = {
    dashboard:'Dashboard', participants:'Teilnehmer', teams:'Teams', orders:'Aufträge', inbox:'Inbox', chat:'Chat', admin:'Admin Settings'
  };
  el('pageTitle').textContent = titleMap[state.page] || '...';

  const page = el('page');
  page.innerHTML='';

  if (state.page==='dashboard') return renderDashboard(page);
  if (state.page==='participants') return renderParticipants(page);
  if (state.page==='teams') return renderTeams(page);
  if (state.page==='orders') return renderOrders(page);
  if (state.page==='inbox') return renderInbox(page);
  if (state.page==='chat') return renderChat(page);
  if (state.page==='admin') return renderAdmin(page);
}

async function renderDashboard(root){
  root.innerHTML = `
    <div class="row">
      <button class="btn" id="btnNewIncident">Neuer Einsatz</button>
      <button class="btn secondary" id="btnLoad">Daten neu laden</button>
    </div>
    <hr />
    <div class="small">Live-Updates: aktiv (SSE). Letztes Event: ${escapeHtml(String(state.lastEventId))}</div>
  `;

  el('btnLoad').onclick = async ()=>{ await refreshAll(); };
  el('btnNewIncident').onclick = async ()=>{
    await openNewIncidentModal();
  };
}

async function renderParticipants(root){
  try{ requireIncident(); } catch(e){ root.innerHTML = `<div class="small">${escapeHtml(e.message)}</div>`; return; }

  // Phase B: roster-driven view + directory editor
  state._partsTab = state._partsTab || 'checkin';
  const [rosterRes, partsRes, capsRes, dirRes] = await Promise.all([
    api(`/api/incidents/${state.incidentId}/roster`),
    api(`/api/incidents/${state.incidentId}/participants`),
    api(`/api/capabilities`),
    api(`/api/members`),
  ]);
  const roster = rosterRes.roster;
  const parts = partsRes.participants;
  const members = dirRes.members.filter(m=>m.is_active);
  const caps = capsRes.capabilities || [];
  const capKeys = caps.map(c=>c.key);

  root.innerHTML = `
    <div class="row" style="gap:8px">
      <button class="btn ${state._partsTab==='checkin'?'':'secondary'}" id="tabCheckin">Check-In/Out</button>
      <button class="btn ${state._partsTab==='directory'?'':'secondary'}" id="tabDir">Mitgliederverzeichnis</button>
    </div>
    <hr />
    <div id="partsBody"></div>
    <datalist id="capList">${capKeys.map(k=>`<option value="${escapeHtml(k)}"></option>`).join('')}</datalist>
  `;

  el('tabCheckin').onclick = ()=>{ state._partsTab='checkin'; render(); };
  el('tabDir').onclick = ()=>{ state._partsTab='directory'; render(); };

  const body = el('partsBody');
  if (state._partsTab==='directory') {
    body.innerHTML = renderMemberDirectoryHtml();
    wireMemberDirectory(members);
    return;
  }

  body.innerHTML = `
    <div class="row" style="align-items:flex-start">
      <div style="min-width:360px;flex:1">
        <div class="small">Schnell-Suche</div>
        <input id="qRoster" placeholder="Name / Code / Fähigkeit" style="width:100%" />
        <div class="row" style="margin-top:8px">
          <button class="btn" id="btnQuick">Schnell Check-In (neu)</button>
          <button class="btn secondary" id="btnCheckinSel">Auswahl Check-In</button>
        </div>
        <div class="small" style="margin-top:10px">Auswahl</div>
        <select id="selRoster" multiple size="10" style="width:100%"></select>
        <div class="row" style="margin-top:8px">
          <button class="btn danger" id="btnCheckoutAll">Alle Check-Out</button>
        </div>
      </div>
      <div style="flex:2">
        <div class="row" style="justify-content:space-between;align-items:center">
          <div class="small">Teilnehmer (eingecheckt): ${parts.length}</div>
          <label class="small"><input type="checkbox" id="onlyIn" checked /> nur eingecheckt</label>
        </div>
        <div class="list" id="participantList"></div>
      </div>
    </div>
  `;

  // fill roster select
  const sel = el('selRoster');
  const fill = (q='')=>{
    sel.innerHTML='';
    const qq = q.trim().toLowerCase();
    for (const r of roster) {
      const hay = `${r.display_name||''} ${r.short_code||''} ${(r.capabilities||[]).join(' ')}`.toLowerCase();
      if (qq && !hay.includes(qq)) continue;
      const o = document.createElement('option');
      o.value = r.member_id;
      const loanTag = r.loaned_in ? ' (geliehen)' : '';
      o.textContent = `${r.display_name}${r.short_code?` (${r.short_code})`:''}${loanTag} · ${r.status}`;
      sel.appendChild(o);
    }
  };
  fill('');
  el('qRoster').oninput = (e)=>fill(e.target.value);

  el('btnCheckinSel').onclick = async ()=>{
    const ids = Array.from(sel.selectedOptions).map(o=>o.value);
    if (!ids.length) return toast('Hinweis','Bitte Mitglieder auswählen');
    await api(`/api/incidents/${state.incidentId}/checkin`, {method:'POST', body:{member_ids:ids}});
    toast('OK','Eingecheckt');
    await render();
  };

  el('btnQuick').onclick = async ()=>{
    const display_name = prompt('Name (neu anlegen + Check-In):');
    if (!display_name) return;
    const short_code = prompt('Kurzcode (optional):') || '';
    const capStr = prompt(`Fähigkeiten (keys, kommasepariert)\nBekannt: ${capKeys.join(', ')}`) || '';
    const capabilities = capStr.split(',').map(s=>s.trim()).filter(Boolean);
    await api(`/api/incidents/${state.incidentId}/checkin-quick`, {method:'POST', body:{display_name, short_code, capabilities}});
    toast('OK','Neu angelegt & eingecheckt');
    await render();
  };

  el('btnCheckoutAll').onclick = async ()=>{
    const reason = prompt('Checkout-Grund (optional):')||'';
    await api(`/api/incidents/${state.incidentId}/checkout`, {method:'POST', body:{all:true, reason}});
    toast('OK','Alle ausgecheckt');
    await render();
  };

  const list = el('participantList');
  const renderList = ()=>{
    const onlyIn = el('onlyIn').checked;
    list.innerHTML='';
    for (const p of parts) {
      if (onlyIn && p.status!=='in') continue;
      const it = document.createElement('div');
      it.className='item';
      const name = p.display_name || p.pseudonym || '(anonym)';
      const loan = p.loan ? ` · Leihe: ${String(p.loan.from_org_id).slice(0,8)} → ${String(p.loan.to_org_id).slice(0,8)}` : '';
      it.innerHTML = `
        <div class="row" style="justify-content:space-between">
          <div>
            <div class="title">${escapeHtml(name)} <span class="small">${escapeHtml(p.org_short||'')}</span></div>
            <div class="small">Status: ${escapeHtml(p.status)} · In: ${escapeHtml(p.checked_in_at||'')} ${p.checked_out_at?`· Out: ${escapeHtml(p.checked_out_at)}`:''}${escapeHtml(loan)}</div>
            <div class="small">Fähigkeiten: ${(p.capabilities||[]).map(escapeHtml).join(', ')||'—'}</div>
          </div>
          <div class="row">
            <button class="btn secondary" data-mid="${p.member_id}" ${p.status==='out'?'disabled':''}>Einzeln Check-Out</button>
          </div>
        </div>
      `;
      it.querySelector('button').onclick = async ()=>{
        const r = prompt('Grund (optional):')||'';
        await api(`/api/incidents/${state.incidentId}/checkout`, {method:'POST', body:{member_id:p.member_id, reason:r}});
        await render();
      };
      list.appendChild(it);
    }
  };
  renderList();
  el('onlyIn').onchange = renderList;
}

function renderMemberDirectoryHtml(){
  return `
    <div class="row" style="justify-content:space-between;align-items:center">
      <div class="small">Mitglieder deiner Organisation</div>
      <button class="btn" id="btnAddMember">Neues Mitglied</button>
    </div>
    <div class="small" style="margin-top:8px">Suche</div>
    <input id="qMembers" placeholder="Name / Kurzcode / Fähigkeit" style="width:100%" />
    <hr />
    <div class="list" id="memberList"></div>
  `;
}

function wireMemberDirectory(members){
  const list = el('memberList');
  const renderList = (q='')=>{
    const qq = q.trim().toLowerCase();
    list.innerHTML='';
    for (const m of members) {
      const hay = `${m.display_name||''} ${m.short_code||''} ${(m.capabilities||[]).join(' ')}`.toLowerCase();
      if (qq && !hay.includes(qq)) continue;
      const it = document.createElement('div');
      it.className='item';
      const canLoan = !!(state.incidentId || state.currentIncidentId || state.currentEinsatzId || state.einsatzId || state.incident_id || state.current?.incidentId);
      it.innerHTML = `
        <div class="row" style="justify-content:space-between">
          <div>
            <div class="title">${escapeHtml(m.display_name)} <span class="small">${escapeHtml(m.short_code||'')}</span></div>
            <div class="small">Fähigkeiten: ${(m.capabilities||[]).map(escapeHtml).join(', ')||'—'}</div>
          </div>
          <div class="row">
            ${canLoan?'<button class="btn">Verleihen</button>':''}
            <button class="btn secondary">Bearbeiten</button>
          </div>
        </div>
      `;
      const btns = it.querySelectorAll('button');
      if (canLoan) {
        btns[0].onclick = ()=>openMemberLoanModal(m.id);
        btns[1].onclick = ()=>openMemberEditor(m);
      } else {
        btns[0].onclick = ()=>openMemberEditor(m);
      }
      list.appendChild(it);
    }
    if (!list.children.length) list.innerHTML = `<div class="small">Keine Treffer.</div>`;
  };
  renderList('');
  el('qMembers').oninput = (e)=>renderList(e.target.value);
  el('btnAddMember').onclick = ()=>openMemberEditor(null);
}

function openMemberEditor(member){
  const name = prompt('Name:', member?.display_name || '');
  if (name===null) return;
  if (!name.trim()) return toast('Fehler','Name erforderlich');
  const short_code = prompt('Kurzcode (optional):', member?.short_code || '') || '';
  const capStr = prompt('Fähigkeiten (keys, kommasepariert):', (member?.capabilities||[]).join(', ')) || '';
  const capabilities = capStr.split(',').map(s=>s.trim()).filter(Boolean);
  api('/api/members', {method:'POST', body:{id: member?.id, display_name:name.trim(), short_code:short_code.trim(), capabilities}})
    .then(()=>{ toast('OK','Gespeichert'); render(); })
    .catch(e=>toast('Fehler', e.message));
}

async function renderTeams(root){
  try{ requireIncident(); } catch(e){ root.innerHTML = `<div class="small">${escapeHtml(e.message)}</div>`; return; }

  const data = await api(`/api/incidents/${state.incidentId}/teams`);
  const teams = data.teams;
  root.innerHTML = `
    <div class="row">
      <button class="btn" id="btnNewTeam">Neues Team</button>
    </div>
    <hr />
    <div class="list" id="teamList"></div>
  `;

  el('btnNewTeam').onclick = async ()=>{
    const name = prompt('Teamname?');
    if (!name) return;
    const callsign = prompt('Rufname (optional)?')||'';
    await api(`/api/incidents/${state.incidentId}/teams`, {method:'POST', body:{name, callsign}});
    toast('OK','Team erstellt');
    await render();
  };

  const list = el('teamList');
  list.innerHTML='';
  const roleDefs = state.settings?.team?.roles || [];
  const roleLabel = (k)=> (roleDefs.find(r=>r.key===k)?.label) || k;

  for (const t of teams) {
    const it = document.createElement('div');
    it.className='item';
    it.innerHTML = `
      <div class="row" style="justify-content:space-between">
        <div>
          <div class="title">${escapeHtml(t.name)} <span class="small">${escapeHtml(t.org_short||'')}</span></div>
          <div class="small">Status: ${escapeHtml(t.status||'')} · Rufname: ${escapeHtml(t.callsign||'')}</div>
          <div class="small">Mitglieder: ${(t.members||[]).map(m=>{
            const r = roleLabel(m.role_in_team||'member');
            return `${escapeHtml(m.display_name)} (${escapeHtml(r)})`;
          }).join(', ')||'—'}</div>
        </div>
        <div class="row">
          <button class="btn secondary">Mitglieder zuweisen</button>
        </div>
      </div>
    `;
    it.querySelector('button').onclick = ()=>openAssignTeam(t);
    list.appendChild(it);
  }
}

async function openAssignTeam(team){
  requireIncident();
  const [parts, rosterRes] = await Promise.all([
    api(`/api/incidents/${state.incidentId}/participants`),
    api(`/api/incidents/${state.incidentId}/roster`)
  ]);
  const avail = parts.participants.filter(p=>p.status==='in');
  const roleDefs = state.settings?.team?.roles || [{key:'member',label:'Mitglied',sort:90}];
  const roleOpts = roleDefs
    .slice()
    .sort((a,b)=>(Number(a.sort||100)-Number(b.sort||100)))
    .map(r=>`<option value="${escapeHtml(r.key)}">${escapeHtml(r.label)}</option>`)
    .join('');

  const current = new Map();
  for (const m of (team.members||[])) {
    current.set(m.member_id, {role: m.role_in_team||'member', is_lead: !!m.is_lead});
  }

  const rowsHtml = avail.map(p=>{
    const cur = current.get(p.member_id);
    const checked = cur ? 'checked' : '';
    const role = cur?.role || 'member';
    const lead = cur?.is_lead ? 'checked' : '';
    const name = escapeHtml(p.display_name || p.pseudonym || p.member_id);
    const org = escapeHtml(p.org_short||p.org_name||'');
    return `
      <tr>
        <td><input type="checkbox" data-mid="${escapeHtml(p.member_id)}" ${checked}></td>
        <td>${name}<div class="small">${org}</div></td>
        <td><select data-role-for="${escapeHtml(p.member_id)}">${roleOpts}</select></td>
        <td style="text-align:center"><input type="checkbox" data-lead-for="${escapeHtml(p.member_id)}" ${lead}></td>
      </tr>`;
  }).join('');

  modal({
    title: `Team: ${team.name} – Mitglieder & Rollen`,
    okText: 'Speichern',
    html: `
      <div class="small">Nur eingecheckte Teilnehmer sind zuweisbar. Rollen kommen aus Admin → Einstellungen.</div>
      <table class="table" style="width:100%;margin-top:8px">
        <thead><tr><th></th><th>Mitglied</th><th>Rolle</th><th>Lead</th></tr></thead>
        <tbody>${rowsHtml || '<tr><td colspan="4" class="small">Keine eingecheckten Teilnehmer.</td></tr>'}</tbody>
      </table>
    `,
    onOk: async ()=>{
      const members = [];
      document.querySelectorAll('input[type="checkbox"][data-mid]').forEach(cb=>{
        if (!cb.checked) return;
        const mid = cb.getAttribute('data-mid');
        const sel = document.querySelector(`select[data-role-for="${CSS.escape(mid)}"]`);
        const leadCb = document.querySelector(`input[data-lead-for="${CSS.escape(mid)}"]`);
        members.push({member_id: mid, role: sel?.value || 'member', is_lead: !!leadCb?.checked});
      });
      await api(`/api/teams/${team.id}/assign`, {method:'POST', body:{members}});
      toast('OK','Team aktualisiert');
      await render();
    }
  });
}

async function renderOrders(root){
  try{ requireIncident(); } catch(e){ root.innerHTML = `<div class="small">${escapeHtml(e.message)}</div>`; return; }

  const data = await api(`/api/incidents/${state.incidentId}/orders`);
  const orders = data.orders;
  root.innerHTML = `
    <div class="row">
      <button class="btn" id="btnNewOrder">Neuer Auftrag</button>
    </div>
    <hr />
    <div class="list" id="orderList"></div>
  `;

  el('btnNewOrder').onclick = async ()=>{
    const tRes = await api(`/api/incidents/${state.incidentId}/teams`);
    const teams = tRes.teams.filter(t=>t.org_id===state.user.org_id);
    const templates = state.settings?.orderTemplates || [];

    const tplOpts = templates.map(t=>`<option value="${escapeHtml(t.key)}">${escapeHtml(t.label||t.key)}</option>`).join('');
    const teamOpts = [`<option value="">—</option>`].concat(teams.map(t=>`<option value="${escapeHtml(t.id)}">${escapeHtml(t.name)}</option>`)).join('');

    const renderFields = (tplKey)=>{
      const tpl = templates.find(t=>t.key===tplKey) || templates[0];
      const fields = Array.isArray(tpl?.fields)?tpl.fields:[];
      const out = [];
      for (const f of fields) {
        const key = f.key;
        const label = f.label||key;
        const req = f.required ? ' *' : '';
        if (f.type==='textarea') {
          out.push(`<div class="field"><label>${escapeHtml(label)}${req}</label><textarea data-f="${escapeHtml(key)}" style="width:100%;min-height:70px"></textarea></div>`);
        } else if (f.type==='select') {
          const opts = (f.options||[]).map(o=>`<option value="${escapeHtml(String(o))}">${escapeHtml(String(o))}</option>`).join('');
          out.push(`<div class="field"><label>${escapeHtml(label)}${req}</label><select data-f="${escapeHtml(key)}" style="width:100%"><option value=""></option>${opts}</select></div>`);
        } else {
          out.push(`<div class="field"><label>${escapeHtml(label)}${req}</label><input data-f="${escapeHtml(key)}" style="width:100%" /></div>`);
        }
      }
      return out.join('');
    };

    modal({
      title:'Neuer Auftrag',
      okText:'Erstellen',
      html:`
        <div class="split">
          <div>
            <div class="field"><label>Template</label><select id="o_tpl" style="width:100%">${tplOpts}</select></div>
            <div class="field"><label>Titel</label><input id="o_title" style="width:100%" placeholder="(optional, sonst Template-Name)"/></div>
            <div class="row"><div class="field" style="flex:1"><label>Priorität</label><input id="o_prio" type="number" min="1" max="5" value="3" style="width:100%"/></div>
              <div class="field" style="flex:2"><label>Team (optional)</label><select id="o_team" style="width:100%">${teamOpts}</select></div></div>
          </div>
          <div>
            <div class="small">Felder aus Template</div>
            <div id="o_fields" style="margin-top:6px">${renderFields(templates[0]?.key||'')}</div>
          </div>
        </div>
      `,
      onOk: async ()=>{
        const template_key = document.getElementById('o_tpl').value;
        const title = document.getElementById('o_title').value.trim();
        const priority = Number(document.getElementById('o_prio').value||3);
        const assigned_team_id = document.getElementById('o_team').value;
        const fields = {};
        document.querySelectorAll('#o_fields [data-f]').forEach(n=>{
          const k = n.getAttribute('data-f');
          const v = (n.value||'').trim();
          if (v!=='') fields[k]=v;
        });
        await api(`/api/incidents/${state.incidentId}/orders`, {method:'POST', body:{template_key, title, priority, assigned_team_id, fields}});
        toast('OK','Auftrag erstellt');
        await render();
      }
    });
    // wire template change
    setTimeout(()=>{
      const sel = document.getElementById('o_tpl');
      if (!sel) return;
      sel.onchange = ()=>{ document.getElementById('o_fields').innerHTML = renderFields(sel.value); };
    },0);
  };

  const list = el('orderList');
  list.innerHTML='';
  for (const o of orders) {
    const it = document.createElement('div');
    it.className='item';

    let transferLine = '';
    if (o.transfer) {
      const st = o.transfer.state || '';
      const mode = o.transfer.mode==='fast' ? 'Fast' : 'Angebot';
      const fromN = o.transfer.from_org_name || '';
      const toN = o.transfer.to_org_name || '';
      const when = o.transfer.decided_at || o.transfer.created_at || '';
      transferLine = `<div class="small">Transfer: ${escapeHtml(st)} · ${escapeHtml(mode)} · ${escapeHtml(fromN)} → ${escapeHtml(toN)} · ${escapeHtml(when)}</div>`;
    }

    it.innerHTML = `
      <div class="row" style="justify-content:space-between;align-items:flex-start">
        <div>
          <div class="title">${escapeHtml(o.title)} <span class="small">${escapeHtml(o.status)}</span></div>
          <div class="small">Org: ${escapeHtml(o.assigned_org_name||'')} · Team: ${escapeHtml(o.team_name||'—')} · Prio: ${escapeHtml(String(o.priority))}</div>
          ${transferLine}
        </div>
        <div class="row">
          <button class="btn secondary" data-a="note">Notiz</button>
          <button class="btn secondary" data-a="transfer">Übergeben</button>
          <button class="btn" data-a="next">Next Status</button>
        </div>
      </div>
    `;
    const btns = it.querySelectorAll('button');
    btns.forEach(b=>{
      b.onclick = async ()=>{
        const a = b.dataset.a;
        if (a==='note') {
          const m = prompt('Notiz:'); if (!m) return;
          await api(`/api/orders/${o.id}/note`, {method:'POST', body:{message:m}});
          return toast('OK','Notiz gespeichert');
        }
        if (a === 'transfer') {
          openOrderTransferModal(o.id);
          return;
        }
        if (a==='next') {
          const next = ({created:'accepted', accepted:'started', started:'completed', completed:'completed'})[o.status] || 'accepted';
          await api(`/api/orders/${o.id}`, {method:'PUT', body:{status:next}});
          toast('OK',`Status -> ${next}`);
          await render();
        }
      }
    });
    list.appendChild(it);
  }
}

async function openOrderTransferModal(orderId) {
  const incidentId =
    state.incidentId ||
    state.currentIncidentId ||
    state.currentEinsatzId ||
    state.einsatzId ||
    state.incident_id ||
    state.current?.incidentId ||
    null;

  if (!incidentId) {
    alert('Kein Einsatz gewählt');
    return;
  }

  const inc = await api(`/api/incidents/${incidentId}`);
  // Logged-in user is stored as state.user in this codebase
  const myOrgId = state.user?.org_id || null;
  const orgs = (inc.orgs || []).filter(o => !myOrgId || o.id !== myOrgId);

  if (orgs.length === 0) {
    alert('Im Einsatz sind keine weiteren Organisationen vorhanden.');
    return;
  }

  const options = orgs.map(o =>
    `<option value="${o.id}">
      ${escapeHtml(o.short_name ? `${o.short_name} – ${o.name}` : o.name)}
    </option>`
  ).join('');

  const defMode = state.settings?.ops?.transferModeDefault || 'offer';
  const grace = Number(state.settings?.ops?.fastModeGraceSeconds ?? 120);

  modal({
    title: 'Auftrag übergeben',
    okText: 'Senden',
    cancelText: 'Abbrechen',
    html: `
      <div class="field">
        <label>Ziel-Organisation</label>
        <select id="transfer_to_org">
          ${options}
        </select>
      </div>
      <div class="field">
        <label>Modus</label>
        <select id="transfer_mode">
          <option value="offer" ${defMode==='offer'?'selected':''}>Angebot (Annehmen/Ablehnen)</option>
          <option value="fast" ${defMode==='fast'?'selected':''}>Fast (auto nach ${grace}s)</option>
        </select>
      </div>
      <div class="hint">
        Fast verhindert einen Flaschenhals: wenn niemand reagiert, wird automatisch angenommen.
      </div>
    `,
    onOk: async () => {
      const sel = document.getElementById('transfer_to_org');
      const to_org_id = sel ? sel.value : '';
      const mode = (document.getElementById('transfer_mode')?.value || 'offer');

      if (!to_org_id) {
        toast('Fehler', 'Bitte Ziel-Organisation auswählen');
        return false; // falls dein modal() false akzeptiert, um offen zu bleiben
      }

      await api(`/api/incidents/${incidentId}/transfers/order`, {
        method: 'POST',
        body: { order_id: orderId, to_org_id, mode }
      });

      toast('OK', mode==='fast' ? 'Transfer (Fast) gesendet' : 'Transfer angeboten');
    }
  });
}

async function openMemberLoanModal(memberId) {
  const incidentId =
    state.incidentId ||
    state.currentIncidentId ||
    state.currentEinsatzId ||
    state.einsatzId ||
    state.incident_id ||
    state.current?.incidentId ||
    null;

  if (!incidentId) {
    alert('Kein Einsatz gewählt');
    return;
  }

  const inc = await api(`/api/incidents/${incidentId}`);
  const myOrgId = state.user?.org_id || null;
  const orgs = (inc.orgs || []).filter(o => !myOrgId || o.id !== myOrgId);

  if (orgs.length === 0) {
    alert('Im Einsatz sind keine weiteren Organisationen vorhanden.');
    return;
  }

  const options = orgs.map(o =>
    `<option value="${o.id}">${escapeHtml(o.short_name ? `${o.short_name} – ${o.name}` : o.name)}</option>`
  ).join('');

  const defMode = state.settings?.ops?.transferModeDefault || 'offer';
  const grace = Number(state.settings?.ops?.fastModeGraceSeconds ?? 120);

  modal({
    title: 'Mitglied verleihen',
    okText: 'Senden',
    cancelText: 'Abbrechen',
    html: `
      <div class="field">
        <label>Ziel-Organisation</label>
        <select id="loan_to_org">${options}</select>
      </div>
      <div class="field">
        <label>Modus</label>
        <select id="loan_mode">
          <option value="offer" ${defMode==='offer'?'selected':''}>Angebot (Annehmen/Ablehnen)</option>
          <option value="fast" ${defMode==='fast'?'selected':''}>Fast (auto nach ${grace}s)</option>
        </select>
      </div>
      <div class="hint">Die Ziel-Organisation kann das Mitglied für Teams im Einsatz nutzen (nach Annahme).</div>
    `,
    onOk: async ()=>{
      const to_org_id = (document.getElementById('loan_to_org')?.value || '');
      const mode = (document.getElementById('loan_mode')?.value || 'offer');
      if (!to_org_id) { toast('Fehler','Bitte Ziel-Organisation auswählen'); return false; }

      await api(`/api/incidents/${incidentId}/loans/member`, {
        method:'POST',
        body:{ member_id: memberId, to_org_id, mode }
      });
      toast('OK', mode==='fast' ? 'Leihe (Fast) gesendet' : 'Leihe angeboten');
      await render();
    }
  });
}

async function renderInbox(root){
  // Scope inbox to the currently selected incident when possible
  const q = state.incidentId ? `?incident_id=${encodeURIComponent(state.incidentId)}` : '';
  const data = await api('/api/inbox' + q);
  const inbox = data.inbox || [];
  const outbox = data.outbox || [];
  const inboxHist = data.inbox_history || [];
  const outboxHist = data.outbox_history || [];
  const loans = data.active_loans || [];
  const tab = state._inboxTab || 'in';

  root.innerHTML = `
    <div class="row" style="gap:8px;flex-wrap:wrap">
      <button class="btn ${tab==='in'?'':'secondary'}" id="tabInbox">Eingang (${inbox.length})</button>
      <button class="btn ${tab==='out'?'':'secondary'}" id="tabOutbox">Ausgang (${outbox.length})</button>
      <button class="btn ${tab==='hist'?'':'secondary'}" id="tabHistory">Historie (${inboxHist.length + outboxHist.length})</button>
      <button class="btn ${tab==='loans'?'':'secondary'}" id="tabLoans">Leihen (${loans.length})</button>
      <div class="small" style="margin-left:auto">${state.incidentId ? 'Gefiltert: aktueller Einsatz' : 'Alle Einsätze'}</div>
    </div>
    <hr />
    <div class="list" id="inboxList"></div>
  `;

  el('tabInbox').onclick = ()=>{ state._inboxTab='in'; render(); };
  el('tabOutbox').onclick = ()=>{ state._inboxTab='out'; render(); };
  el('tabHistory').onclick = ()=>{ state._inboxTab='hist'; render(); };
  el('tabLoans').onclick = ()=>{ state._inboxTab='loans'; render(); };

  const list = el('inboxList');
  list.innerHTML = '';

  // Pick dataset depending on tab
  let rows = [];
  if (tab==='out') rows = outbox;
  else if (tab==='in') rows = inbox;
  else if (tab==='loans') rows = loans;
  else {
    // History: merge incoming + outgoing with a direction hint
    const merged = [];
    for (const r of outboxHist) merged.push({...r, _dir:'out'});
    for (const r of inboxHist) merged.push({...r, _dir:'in'});
    merged.sort((a,b)=>{
      const ta = Date.parse(a.decided_at || a.created_at || 0) || 0;
      const tb = Date.parse(b.decided_at || b.created_at || 0) || 0;
      return tb - ta;
    });
    rows = merged;
  }

  if (!rows.length) {
    const msg = tab==='out' ? 'Keine offenen ausgehenden Angebote.'
      : tab==='in' ? 'Keine offenen eingehenden Angebote.'
      : tab==='hist' ? 'Keine Historie vorhanden.'
      : 'Keine aktiven Leihen.';
    list.innerHTML = `<div class="small">${escapeHtml(msg)}</div>`;
    return;
  }

  for (const r of rows) {
    const it = document.createElement('div');
    it.className='item';
    const isLoanRow = (tab==='loans');
    const isHist = (tab==='hist');

    let what = '';
    let peer = '';
    let meta = '';
    let actionsHtml = '';

    if (isLoanRow) {
      what = `Mitglied (Leihe): ${r.member_name || ''}`;
      peer = `Von: ${r.from_org_name} · An: ${r.to_org_name}`;
      meta = `Seit: ${r.started_at}`;

      const myOrg = state.user?.org_id;
      const canEnd = myOrg && (r.to_org_id===myOrg || r.from_org_id===myOrg) || state.user?.role==='admin';
      actionsHtml = canEnd ? `<button class="btn danger" data-act="endloan">Zurückgeben</button>` : '';
    } else {
      const payload = r.payload || {};
      what = r.type==='order_transfer' ? `Auftrag: ${payload.order_title || ''}` : `Mitglied: ${payload.member_name || ''}`;
      const dir = isHist ? (r._dir||'') : tab;
      peer = dir==='out' ? `An: ${r.to_org_name}` : `Von: ${r.from_org_name}`;

      const exp = r.expires_at ? ` · Ablauf: ${r.expires_at}` : '';
      const mode = r.mode==='fast' ? 'Fast' : 'Angebot';
      const st = r.state ? ` · Status: ${r.state}` : '';
      const when = isHist ? ` · Entscheiden: ${(r.decided_at || r.created_at || '')}` : '';
      meta = `Modus: ${mode}${exp}${st}${when}`;

      if (!isHist) {
        if (tab === 'in') {
          actionsHtml = `
            <button class="btn" data-act="accept">Annehmen</button>
            <button class="btn danger" data-act="reject">Ablehnen</button>
          `;
        } else {
          actionsHtml = `
            <button class="btn danger" data-act="cancel">Abbrechen</button>
          `;
        }
      }
    }

    it.innerHTML = `
      <div class="row" style="justify-content:space-between">
        <div>
          <div class="title">${escapeHtml(what)}</div>
          <div class="small">${escapeHtml(peer)}${meta ? ' · ' + escapeHtml(meta) : ''}</div>
        </div>
        <div class="row" style="gap:8px">
          ${actionsHtml}
        </div>
      </div>
    `;

    if (tab === 'loans') {
      const btn = it.querySelector('[data-act="endloan"]');
      if (btn) {
        btn.onclick = async ()=>{
          if (!confirm('Leihe wirklich beenden / zurückgeben?')) return;
          await api(`/api/loans/${r.id}/end`, {method:'POST', body:{}});
          toast('OK','Leihe beendet');
          await render();
        };
      }
    } else if (tab === 'in') {
      it.querySelector('[data-act="accept"]').onclick = async ()=>{
        await api(`/api/transfer/${r.id}/decide`, {method:'POST', body:{decision:'accept'}});
        toast('OK','Angenommen');
        await render();
      };
      it.querySelector('[data-act="reject"]').onclick = async ()=>{
        await api(`/api/transfer/${r.id}/decide`, {method:'POST', body:{decision:'reject'}});
        toast('OK','Abgelehnt');
        await render();
      };
    } else if (tab === 'out') {
      it.querySelector('[data-act="cancel"]').onclick = async ()=>{
        if (!confirm('Angebot wirklich abbrechen?')) return;
        await api(`/api/transfer/${r.id}/cancel`, {method:'POST', body:{}});
        toast('OK','Abgebrochen');
        await render();
      };
    }

    list.appendChild(it);
  }
}

async function renderChat(root){
  try{ requireIncident(); } catch(e){ root.innerHTML = `<div class="small">${escapeHtml(e.message)}</div>`; return; }

  // load targets once
  if (!state.chatTargets[state.incidentId]) {
    const t = await api(`/api/incidents/${state.incidentId}/chat/targets`);
    state.chatTargets[state.incidentId] = { orgs: t.orgs || [], users: t.users || [] };
  }
  const targets = state.chatTargets[state.incidentId];

  // Load pinned messages and latest chat messages
  await chatLoadPins(state.incidentId);
  const data = await api(`/api/incidents/${state.incidentId}/chat?limit=200`);
  chatCacheMerge(state.incidentId, data.messages || []);

  state.chatUi = state.chatUi || {};
  if (!state.chatUi[state.incidentId]) state.chatUi[state.incidentId] = { onlyMentions:false, search:'' };
  const ui = state.chatUi[state.incidentId];

  root.innerHTML = `
    <div class="row" style="justify-content:space-between; align-items:center; gap:8px">
      <div class="row" style="gap:8px">
        <button class="btn secondary" id="chatBtnMentions">${ui.onlyMentions ? 'Alle Nachrichten' : 'Nur Mentions'}</button>
        <button class="btn secondary" id="chatBtnJump">Letzte Mention</button>
      </div>
      <input id="chatSearch" placeholder="Suche..." value="${escapeHtml(ui.search||'')}" style="max-width:240px" />
    </div>
    <div id="chatPins" class="chat-pins" style="display:none"></div>
    <div class="chat-thread" id="chatList" style="max-height:520px;overflow:auto;margin-top:8px"></div>
    <hr />
    <div class="row" style="gap:8px; align-items:flex-start; position:relative">
      <select id="chatScope" title="Sichtbarkeit">
        <option value="public">Öffentlich (Einsatz)</option>
        <option value="org">Eigene Org</option>
        <option value="org_other">Andere Org…</option>
        <option value="gfs">GFs/EL</option>
        <option value="all">Alle (cross-org)</option>
        <option value="private">Privat…</option>
      </select>
      <select id="chatToOrg" style="display:none"></select>
      <select id="chatToUser" style="display:none"></select>
      <div id="chatScopeHint" class="chat-scope-hint" style="display:none"></div>
      <div style="flex:1; position:relative">
        <textarea id="chatMsg" placeholder="Nachricht... (@mention)" style="width:100%; min-height:44px; resize:vertical" autocomplete="off"></textarea>
        <div id="mentionPicker" class="mention-picker" style="display:none"></div>
      </div>
      <button class="btn" id="btnSend">Senden</button>
    </div>
    <div class="row" style="justify-content:space-between; margin-top:6px">
      <div class="small">Mentions: @all, @gfs, @org:KURZ, @user:username</div>
      <div class="small">Enter = senden · Shift+Enter = Zeilenumbruch</div>
    </div>
  `;

  // render messages (deduped & cached)
  const list = el('chatList');
  chatRenderPins(state.incidentId);
  chatRenderList(state.incidentId, list);
  // When opening chat, jump to newest message so users always see new arrivals
  list.scrollTop = list.scrollHeight;

  // filter/search controls
  el('chatBtnMentions').onclick = ()=>{
    ui.onlyMentions = !ui.onlyMentions;
    state.chatUi[state.incidentId] = ui;
    el('chatBtnMentions').textContent = ui.onlyMentions ? 'Alle Nachrichten' : 'Nur Mentions';
    chatRenderList(state.incidentId, list);
  };
  el('chatSearch').oninput = (e)=>{
    ui.search = String(e.target.value||'');
    state.chatUi[state.incidentId] = ui;
    chatRenderList(state.incidentId, list);
  };
  el('chatBtnJump').onclick = ()=>{
    const items = Array.from(list.querySelectorAll('.chat-msg.mention-hit'));
    if (!items.length) return toast('Hinweis','Keine Mentions gefunden');
    items[items.length-1].scrollIntoView({block:'center'});
  };

  // mark as read
  clearChatUnread(state.incidentId);
  buildNav();
  setChatLastSeen(state.incidentId, new Date().toISOString());

  // wire scope dropdowns
  const scopeSel = el('chatScope');
  const orgSel = el('chatToOrg');
  const userSel = el('chatToUser');
  const scopeHint = el('chatScopeHint');
  orgSel.innerHTML = targets.orgs
    .filter(o => o.id !== state.user?.org_id)
    .map(o => `<option value="${o.id}">${escapeHtml(o.short_name?`${o.short_name} – ${o.name}`:o.name)}</option>`)
    .join('');
  userSel.innerHTML = targets.users
    .filter(u => String(u.id) !== String(state.user?.id))
    .map(u => `<option value="${u.id}">${escapeHtml(`${u.org_short||''} · ${u.display_name||u.username}`)}</option>`)
    .join('');

  function applyScopeUI(){
    const v = scopeSel.value;
    orgSel.style.display = (v === 'org_other') ? '' : 'none';
    userSel.style.display = (v === 'private') ? '' : 'none';
    // show a small hint badge for explicit targets
    let hint = '';
    if (v === 'org_other') {
      const opt = orgSel.options[orgSel.selectedIndex];
      hint = opt ? `An Org: ${opt.textContent}` : '';
    } else if (v === 'private') {
      const opt = userSel.options[userSel.selectedIndex];
      hint = opt ? `Privat an: ${opt.textContent}` : '';
    }
    if (hint) {
      scopeHint.textContent = hint;
      scopeHint.style.display = '';
    } else {
      scopeHint.textContent = '';
      scopeHint.style.display = 'none';
    }
  }
  scopeSel.onchange = applyScopeUI;
  orgSel.onchange = applyScopeUI;
  userSel.onchange = applyScopeUI;
  applyScopeUI();

  // mention picker
  const inp = el('chatMsg');
  const picker = el('mentionPicker');
  let pickIdx = 0;
  function buildSuggestions(q){
    const qq = (q||'').toLowerCase();
    const items = [];
    items.push({label:'@all', insert:'@all'});
    items.push({label:'@gfs', insert:'@gfs'});
    for (const o of targets.orgs) {
      if (!o.short_name) continue;
      const ins = `@org:${o.short_name}`;
      const lbl = `${ins} (${o.name})`;
      if (!qq || lbl.toLowerCase().includes(qq)) items.push({label:lbl, insert:ins});
    }
    for (const u of targets.users) {
      const uname = u.username || '';
      const ins = `@user:${uname}`;
      const lbl = `${ins} (${u.display_name||uname})`;
      if (!qq || lbl.toLowerCase().includes(qq)) items.push({label:lbl, insert:ins});
    }
    return items.slice(0, 12);
  }
  function showPicker(items){
    if (!items.length) { picker.style.display='none'; picker.innerHTML=''; return; }
    picker.innerHTML = items.map((it,i)=>`<div class="mi ${i===pickIdx?'active':''}" data-i="${i}">${escapeHtml(it.label)}</div>`).join('');
    picker.style.display='block';
  }
  function hidePicker(){ picker.style.display='none'; picker.innerHTML=''; }
  function getAtContext(text, pos){
    const before = text.slice(0, pos);
    const at = before.lastIndexOf('@');
    if (at === -1) return null;
    // stop if whitespace between @ and cursor
    const frag = before.slice(at+1);
    if (frag.includes(' ') || frag.includes('\n')) return null;
    return {at, q: frag};
  }

  let currentItems = [];
  inp.addEventListener('input', ()=>{
    const ctx = getAtContext(inp.value, inp.selectionStart||0);
    if (!ctx) return hidePicker();
    pickIdx = 0;
    currentItems = buildSuggestions(ctx.q);
    showPicker(currentItems);
  });
  inp.addEventListener('keydown', (e)=>{
    // If picker open, handle navigation/selection
    if (picker.style.display === 'block') {
      if (e.key === 'ArrowDown') { e.preventDefault(); pickIdx = Math.min(pickIdx+1, currentItems.length-1); showPicker(currentItems); return; }
      if (e.key === 'ArrowUp') { e.preventDefault(); pickIdx = Math.max(pickIdx-1, 0); showPicker(currentItems); return; }
      if (e.key === 'Enter') {
        e.preventDefault();
        const ctx = getAtContext(inp.value, inp.selectionStart||0);
        if (!ctx) return;
        const it = currentItems[pickIdx];
        if (!it) return;
        const cur = inp.selectionStart||0;
        inp.value = inp.value.slice(0, ctx.at) + it.insert + ' ' + inp.value.slice(cur);
        hidePicker();
        return;
      }
      if (e.key === 'Escape') { hidePicker(); return; }
    }

    // Sending: Enter = send, Shift+Enter = newline
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      el('btnSend').click();
    }
  });
  picker.addEventListener('mousedown', (e)=>{
    const row = e.target.closest('.mi');
    if (!row) return;
    e.preventDefault();
    const i = Number(row.getAttribute('data-i')||0);
    const it = currentItems[i];
    const ctx = getAtContext(inp.value, inp.selectionStart||0);
    if (!ctx || !it) return;
    inp.value = inp.value.slice(0, ctx.at) + it.insert + ' ' + inp.value.slice(inp.selectionStart||0);
    hidePicker();
    inp.focus();
  });

  el('btnSend').onclick = async ()=>{
    const message = el('chatMsg').value.trim();
    if (!message) return;
    const scopeRaw = el('chatScope').value;
    let scope = scopeRaw;
    const body = { message, scope: 'public' };
    if (scopeRaw === 'org') {
      body.scope = 'org';
    } else if (scopeRaw === 'org_other') {
      body.scope = 'org';
      body.to_org_id = orgSel.value;
    } else if (scopeRaw === 'gfs') {
      body.scope = 'gfs';
    } else if (scopeRaw === 'all') {
      body.scope = 'all';
    } else if (scopeRaw === 'private') {
      body.scope = 'private';
      body.to_user_id = userSel.value;
    } else {
      body.scope = 'public';
    }
    body.mentions = parseMentions(message);

    await api(`/api/incidents/${state.incidentId}/chat`, {method:'POST', body});
    el('chatMsg').value='';
    hidePicker();
    // Pull only delta and update list to avoid duplicate renders/races
    await chatPullAndUpdate(state.incidentId, el('chatList'));
    clearChatUnread(state.incidentId);
    buildNav();
    setChatLastSeen(state.incidentId, new Date().toISOString());
  };

}

async function renderAdmin(root){
  const adminTab = localStorage.getItem('adminTab') || 'settings';
  const tabs = [
    ['settings','Einstellungen'],
    ['templates','Auftrags-Templates'],
    ['orgs','Organisationen'],
    ['users','Benutzer'],
  ];
  root.innerHTML = `
    <div class="row" id="adminTabs"></div>
    <hr />
    <div id="adminBody"></div>
  `;
  const tabWrap = root.querySelector('#adminTabs');
  for (const [k, label] of tabs) {
    const b = document.createElement('button');
    b.className = 'btn secondary';
    if (k === adminTab) b.className = 'btn';
    b.textContent = label;
    b.onclick = ()=>{ localStorage.setItem('adminTab', k); render(); };
    tabWrap.appendChild(b);
  }
  const body = root.querySelector('#adminBody');

  if (adminTab === 'settings') return renderAdminSettings(body);
  if (adminTab === 'templates') return renderAdminTemplates(body);
  if (adminTab === 'orgs') return renderAdminOrgs(body);
  if (adminTab === 'users') return renderAdminUsers(body);
}

async function renderAdminSettings(root){
  const rows = (await api('/api/settings')).settings;
  const map = Object.fromEntries(rows.map(r=>[r.key, r.value]));
  const branding = map.branding || {};
  const privacy = map.privacy || {};
  const ops = map.ops || {};
  const team = map.team || {};
  const roles = Array.isArray(team.roles) ? team.roles : [];
  const rolesText = roles
    .sort((a,b)=>(Number(a.sort||100)-Number(b.sort||100)))
    .map(r=>`${r.key||''}|${r.label||''}|${r.sort??''}`)
    .join('\n');

  root.innerHTML = `
    <div class="split">
      <div>
        <div class="small">Branding</div>
        <div class="row"><input id="s_appName" placeholder="Tool-Name" style="flex:1" value="${escapeHtml(branding.appName||'')}" /></div>
        <div class="row"><input id="s_primary" placeholder="#Primary" value="${escapeHtml(branding.primaryColor||'')}" />
          <input id="s_accent" placeholder="#Accent" value="${escapeHtml(branding.accentColor||'')}" /></div>
        <div class="row"><button class="btn secondary" id="btnLogo">Logo upload</button><button class="btn secondary" id="btnFav">Favicon upload</button></div>
        <div class="small">Logo: ${escapeHtml(branding.logoPath||'—')}</div>
        <div class="small">Favicon: ${escapeHtml(branding.faviconPath||'—')}</div>
      </div>
      <div>
        <div class="small">Datenschutz</div>
        <label class="small"><input type="checkbox" id="s_pseudo" ${privacy.pseudonymizeOtherOrgs?'checked':''}/> Andere Orgs pseudonymisieren</label><br/>
        <label class="small"><input type="checkbox" id="s_showNames" ${privacy.showNamesToOtherOrgs?'checked':''}/> Klarnamen für andere Orgs anzeigen</label><br/>
        <div class="row"><input id="s_retention" type="number" min="0" value="${escapeHtml(String(privacy.retentionDays??180))}" /> <span class="small">Retention (Tage)</span></div>
        <hr />
        <div class="small">Betrieb</div>
        <div class="row"><select id="s_transferMode"><option value="offer">offer</option><option value="fast">fast</option></select>
          <input id="s_fastGrace" type="number" min="0" value="${escapeHtml(String(ops.fastModeGraceSeconds??120))}" /> <span class="small">Fast-Grace (sek)</span></div>
        <div class="row">
          <input id="s_hbWarn" type="number" min="0" value="${escapeHtml(String(ops.heartbeatWarnMinutes??30))}" /> <span class="small">Heartbeat Warn (min)</span>
        </div>
        <div class="row">
          <input id="s_hbAlarm" type="number" min="0" value="${escapeHtml(String(ops.heartbeatAlarmMinutes??45))}" /> <span class="small">Heartbeat Alarm (min)</span>
        </div>

        <hr />
        <div class="small">Team-Rollen (frei definierbar)</div>
        <div class="small">Format pro Zeile: <code>key|Label|Sort</code> (Sort optional)</div>
        <textarea id="s_teamRoles" style="width:100%;min-height:120px" placeholder="lead|Leitung|10\nnav|Navigation|20\nradio|Funk|30\nmember|Mitglied|90">${escapeHtml(rolesText)}</textarea>
        <label class="small"><input type="checkbox" id="s_multiLead" ${team.allowMultipleLeads!==false?'checked':''}/> Mehrere Leitungs-Rollen pro Team erlauben</label>
      </div>
    </div>
    <hr />
    <button class="btn" id="btnSaveSettings">Speichern</button>
  `;
  el('s_transferMode').value = ops.transferModeDefault || 'offer';

  el('btnSaveSettings').onclick = async ()=>{
    const newBrand = {
      appName: el('s_appName').value.trim(),
      primaryColor: el('s_primary').value.trim(),
      accentColor: el('s_accent').value.trim(),
      logoPath: branding.logoPath || null,
      faviconPath: branding.faviconPath || null,
    };
    const newPrivacy = {
      pseudonymizeOtherOrgs: el('s_pseudo').checked,
      showNamesToOtherOrgs: el('s_showNames').checked,
      retentionDays: Number(el('s_retention').value||0),
    };
    const newOps = {
      transferModeDefault: el('s_transferMode').value,
      fastModeGraceSeconds: Number(el('s_fastGrace').value||0),
      heartbeatWarnMinutes: Number(el('s_hbWarn').value||0),
      heartbeatAlarmMinutes: Number(el('s_hbAlarm').value||0),
    };
    const roleLines = (el('s_teamRoles').value||'').split(/\r?\n/).map(s=>s.trim()).filter(Boolean);
    const newRoles = [];
    for (const line of roleLines) {
      const [k,l,srt] = line.split('|').map(x=>String(x||'').trim());
      if (!k || !l) continue;
      newRoles.push({key:k, label:l, sort: srt!=='' ? Number(srt) : 100});
    }
    const newTeam = {
      roles: newRoles.length ? newRoles : roles,
      allowMultipleLeads: el('s_multiLead').checked,
    };
    await api('/api/settings', {method:'POST', body:{key:'branding', value:newBrand}});
    await api('/api/settings', {method:'POST', body:{key:'privacy', value:newPrivacy}});
    await api('/api/settings', {method:'POST', body:{key:'ops', value:newOps}});
    await api('/api/settings', {method:'POST', body:{key:'team', value:newTeam}});
    toast('OK','Einstellungen gespeichert');
    const pub = await api('/api/public-settings');
    applyBrand(pub.settings);
  };

  const doUpload = async (key)=>{
    const inp = document.createElement('input');
    inp.type='file';
    inp.accept='.png,.jpg,.jpeg,.svg,.ico,.webp';
    inp.onchange = async ()=>{
      if (!inp.files?.[0]) return;
      const fd = new FormData();
      fd.append('file', inp.files[0]);
      fd.append('key', key);
      await api('/api/settings/upload', {method:'POST', body:fd, isForm:true});
      toast('OK','Upload gespeichert');
      await render();
    };
    inp.click();
  };
  el('btnLogo').onclick = ()=>doUpload('branding.logoPath');
  el('btnFav').onclick = ()=>doUpload('branding.faviconPath');
}

async function renderAdminTemplates(root){
  const rows = (await api('/api/settings')).settings;
  const map = Object.fromEntries(rows.map(r=>[r.key, r.value]));
  const templates = map.orderTemplates || [];

  root.innerHTML = `
    <div class="row" style="justify-content:space-between">
      <div class="small">Templates werden global gespeichert (Settings-Key: <code>orderTemplates</code>).</div>
      <button class="btn secondary" id="btnAddTpl">+ Template</button>
    </div>
    <hr />
    <div class="list" id="tplList"></div>
    <hr />
    <button class="btn" id="btnSaveTpl">Templates speichern</button>
  `;

  const list = root.querySelector('#tplList');
  const local = JSON.parse(JSON.stringify(templates));

  const renderList = ()=>{
    list.innerHTML='';
    local.forEach((t, idx)=>{
      const it = document.createElement('div');
      it.className='item';
      it.innerHTML = `
        <div class="row" style="justify-content:space-between;gap:10px">
          <div style="flex:1">
            <div class="row">
              <input data-k="key" data-idx="${idx}" placeholder="key" value="${escapeHtml(t.key||'')}" />
              <input data-k="name" data-idx="${idx}" placeholder="Name" style="flex:1" value="${escapeHtml(t.name||'')}" />
            </div>
            <div class="small" style="margin-top:8px">Felder (JSON):</div>
            <textarea data-k="fields" data-idx="${idx}" rows="5" style="width:100%">${escapeHtml(JSON.stringify(t.fields||[], null, 2))}</textarea>
          </div>
          <div class="row" style="flex-direction:column;align-items:flex-end">
            <button class="btn danger" data-del="${idx}">Löschen</button>
          </div>
        </div>
      `;
      it.querySelector('[data-del]').onclick = ()=>{ local.splice(idx,1); renderList(); };
      list.appendChild(it);
    });
  };

  renderList();

  root.querySelector('#btnAddTpl').onclick = ()=>{
    local.unshift({key:'', name:'', fields:[]});
    renderList();
  };

  root.querySelector('#btnSaveTpl').onclick = async ()=>{
    // pull inputs
    const inputs = root.querySelectorAll('input[data-idx], textarea[data-idx]');
    const next = JSON.parse(JSON.stringify(local));
    for (const inp of inputs) {
      const idx = Number(inp.getAttribute('data-idx'));
      const k = inp.getAttribute('data-k');
      if (!next[idx]) continue;
      if (k === 'fields') {
        try{
          next[idx].fields = JSON.parse(inp.value || '[]');
        } catch(e){
          return toast('Fehler', `Ungültiges JSON bei Template #${idx+1}`);
        }
      } else {
        next[idx][k] = inp.value.trim();
      }
    }
    // validate keys unique
    const keys = next.map(t=>t.key).filter(Boolean);
    const uniq = new Set(keys);
    if (uniq.size !== keys.length) return toast('Fehler','Template keys müssen eindeutig sein');
    await api('/api/settings', {method:'POST', body:{key:'orderTemplates', value:next}});
    toast('OK','Templates gespeichert');
  };
}

async function renderAdminOrgs(root){
  const res = await api('/api/admin/orgs');
  const orgs = res.orgs;
  root.innerHTML = `
    <div class="row" style="justify-content:space-between">
      <div class="small">Organisationen (Mandanten)</div>
      <button class="btn" id="btnOrgAdd">+ Organisation</button>
    </div>
    <hr />
    <div class="list" id="orgList"></div>
  `;
  const list = root.querySelector('#orgList');
  const renderRow = (o)=>{
    const it = document.createElement('div');
    it.className='item';
    it.innerHTML = `
      <div class="row" style="justify-content:space-between;gap:10px">
        <div style="flex:1">
          <div class="row">
            <input value="${escapeHtml(o.name||'')}" data-f="name" style="flex:2" />
            <input value="${escapeHtml(o.short_name||'')}" data-f="short_name" placeholder="Kurz" style="width:110px" />
            <input value="${escapeHtml(o.color||'')}" data-f="color" placeholder="#color" style="width:120px" />
          </div>
          <div class="small">ID: ${escapeHtml(o.id)}</div>
        </div>
        <div class="row" style="align-items:center">
          <label class="small"><input type="checkbox" data-f="is_active" ${o.is_active?'checked':''}/> aktiv</label>
          <button class="btn secondary" data-save>Speichern</button>
          <button class="btn danger" data-del ${o.is_active?'':'disabled'}>Deaktivieren</button>
        </div>
      </div>
    `;
    it.querySelector('[data-save]').onclick = async ()=>{
      const payload = {};
      for (const inp of it.querySelectorAll('[data-f]')) {
        const f = inp.getAttribute('data-f');
        payload[f] = inp.type==='checkbox' ? inp.checked : inp.value.trim();
      }
      await api(`/api/admin/orgs/${o.id}`, {method:'PUT', body:payload});
      toast('OK','Organisation gespeichert');
    };
    it.querySelector('[data-del]').onclick = async ()=>{
      if (!confirm('Organisation deaktivieren?')) return;
      await api(`/api/admin/orgs/${o.id}`, {method:'DELETE'});
      toast('OK','Deaktiviert');
      await render();
    };
    return it;
  };
  list.innerHTML='';
  for (const o of orgs) list.appendChild(renderRow(o));

  root.querySelector('#btnOrgAdd').onclick = async ()=>{
    const name = prompt('Name der Organisation?');
    if (!name) return;
    const short_name = prompt('Kurzname (optional)?') || '';
    const color = prompt('Farbe (optional, z.B. #d6002a)?') || '';
    await api('/api/admin/orgs', {method:'POST', body:{name, short_name, color}});
    toast('OK','Organisation erstellt');
    await render();
  };
}

async function renderAdminUsers(root){
  const [orgRes, userRes] = await Promise.all([
    api('/api/admin/orgs'),
    api('/api/admin/users'),
  ]);
  const orgs = orgRes.orgs;
  const users = userRes.users;
  const orgOpts = orgs.map(o=>`<option value="${escapeHtml(o.id)}">${escapeHtml(o.name)}</option>`).join('');

  root.innerHTML = `
    <div class="row" style="justify-content:space-between">
      <div class="small">Benutzer (Rollen: admin / el / gf / viewer)</div>
      <button class="btn" id="btnUserAdd">+ Benutzer</button>
    </div>
    <hr />
    <div class="list" id="userList"></div>
  `;
  const list = root.querySelector('#userList');
  list.innerHTML='';
  for (const u of users) {
    const it = document.createElement('div');
    it.className='item';
    it.innerHTML = `
      <div class="row" style="justify-content:space-between;gap:10px">
        <div style="flex:1">
          <div class="row">
            <div style="min-width:140px"><b>${escapeHtml(u.username)}</b></div>
            <input data-f="display_name" value="${escapeHtml(u.display_name||'')}" placeholder="Display" style="flex:1" />
            <select data-f="role">
              <option value="admin">admin</option>
              <option value="el">el</option>
              <option value="gf">gf</option>
              <option value="viewer">viewer</option>
            </select>
            <select data-f="org_id">${orgOpts}</select>
          </div>
          <div class="small">ID: ${escapeHtml(String(u.id))} · Org: ${escapeHtml(u.org_name)} · Active: ${u.is_active?'yes':'no'}</div>
        </div>
        <div class="row" style="align-items:center">
          <label class="small"><input type="checkbox" data-f="is_active" ${u.is_active?'checked':''}/> aktiv</label>
          <button class="btn secondary" data-save>Speichern</button>
          <button class="btn danger" data-reset>Passwort</button>
        </div>
      </div>
    `;
    it.querySelector('select[data-f="role"]').value = u.role;
    it.querySelector('select[data-f="org_id"]').value = u.org_id;
    it.querySelector('[data-save]').onclick = async ()=>{
      const payload = {};
      for (const inp of it.querySelectorAll('[data-f]')) {
        const f = inp.getAttribute('data-f');
        payload[f] = inp.type==='checkbox' ? inp.checked : inp.value;
      }
      await api(`/api/admin/users/${u.id}`, {method:'PUT', body:payload});
      toast('OK','User gespeichert');
    };
    it.querySelector('[data-reset]').onclick = async ()=>{
      const pw = prompt('Neues Passwort:');
      if (!pw) return;
      await api(`/api/admin/users/${u.id}/reset-password`, {method:'POST', body:{password:pw}});
      toast('OK','Passwort gesetzt');
    };
    list.appendChild(it);
  }

  root.querySelector('#btnUserAdd').onclick = async ()=>{
    const username = prompt('Username?');
    if (!username) return;
    const password = prompt('Start-Passwort?');
    if (!password) return;
    const display_name = prompt('Display Name (optional)?')||'';
    const role = prompt('Rolle (admin/el/gf/viewer)?','viewer')||'viewer';
    const org_id = orgs[0]?.id;
    await api('/api/admin/users', {method:'POST', body:{username, password, display_name, role, org_id}});
    toast('OK','Benutzer erstellt');
    await render();
  };
}

async function refreshAll(){
  await loadIncidents();
  await render();
}

function connectSSE(){
  if (state.sse) state.sse.close();
  const url = `/events?lastEventId=${encodeURIComponent(state.lastEventId)}`;
  const es = new EventSource(url, {withCredentials:true});
  state.sse = es;

  es.onmessage = (ev)=>{
    // default message event (unused)
  };
  es.addEventListener('hello', ()=>{});

  const handlers = {
    'incident.created': async ()=>{ await loadIncidents(); },
    'incident.updated': async ()=>{ await loadIncidents(); },
    'participants.updated': async (ev)=>{ if (state.page==='participants') await render(); },
    'teams.updated': async (ev)=>{ if (state.page==='teams') await render(); },
    'orders.updated': async (ev)=>{ if (state.page==='orders') await render(); },
    'chat.updated': async (ev)=>{
      // If we're currently in chat page, pull only delta and update list (avoids duplicate renders/races).
      if (state.page==='chat') {
        const incidentId = state.incidentId;
        if (!incidentId) return;
        await chatPullAndUpdate(incidentId, el('chatList'), {limit:200});
        clearChatUnread(incidentId);
        buildNav();
        setChatLastSeen(incidentId, new Date().toISOString());
        return;
      }

      // Otherwise: fetch only new messages since lastSeen for this incident
      const incidentId = state.incidentId;
      if (!incidentId) return;
      const since = getChatLastSeen(incidentId);
      const url = since ? `/api/incidents/${incidentId}/chat?since=${encodeURIComponent(since)}` : `/api/incidents/${incidentId}/chat?limit=10`;
      const res = await api(url);
      const msgs = (res.messages || []);
      if (!msgs.length) return;

      // Keep cache warm so opening chat later shows the new messages immediately
      chatCacheMerge(incidentId, msgs);
      incChatUnread(incidentId, msgs.length);
      buildNav();

      // basic highlight: mention detection
      const myUser = state.user?.username || '';
      let myOrgShort = '';
      const tcache = state.chatTargets?.[incidentId];
      if (tcache && state.user?.org_id) {
        const oo = (tcache.orgs || []).find(o => o.id === state.user.org_id);
        if (oo && oo.short_name) myOrgShort = String(oo.short_name).toLowerCase();
      }
      const hit = msgs.some(m => {
        const t = String(m.message||'');
        if (t.includes('@all') || t.includes('@gfs')) return true;
        if (myUser && t.includes(`@user:${myUser}`)) return true;
        if (myOrgShort && t.toLowerCase().includes(`@org:${myOrgShort}`)) return true;
        return false;
      });

      toast(hit ? 'Chat-Mention' : 'Chat', `Neue Nachricht(en): ${msgs.length}`, [
        {label:'Öffnen', onClick: ()=>{ state.page='chat'; buildNav(); render(); }}
      ]);
    },
    'chat.pins.updated': async (ev)=>{
      const incidentId = state.incidentId;
      if (!incidentId) return;
      await chatLoadPins(incidentId);
      if (state.page==='chat') {
        chatRenderPins(incidentId);
        chatRenderList(incidentId, el('chatList'));
      }
    },
    'chat.pins.updated': async (ev)=>{
      const incidentId = state.incidentId;
      if (!incidentId) return;
      await chatLoadPins(incidentId);
      if (state.page==='chat') {
        chatRenderPins(incidentId);
        chatRenderList(incidentId, el('chatList'));
      }
    },
    'inbox.updated': async (ev)=>{ if (state.page==='inbox') await render(); toast('Inbox','Neues Angebot'); },
    'branding.updated': async ()=>{ const pub = await api('/api/public-settings'); applyBrand(pub.settings); },
    'settings.updated': async ()=>{ const pub = await api('/api/public-settings'); applyBrand(pub.settings); },
    'orgs.updated': async ()=>{ if (state.page==='admin') await render(); },
    'users.updated': async ()=>{ if (state.page==='admin') await render(); },
    'member.created': async ()=>{ if (state.page==='participants') await render(); },
    'capabilities.updated': async ()=>{ if (state.page==='participants') await render(); if (state.page==='admin') await render(); },
    'transfer.decided': async (ev)=>{ toast('Transfer', 'Entscheidung eingegangen'); },
  };

  for (const [type, fn] of Object.entries(handlers)) {
    es.addEventListener(type, async (ev)=>{
      if (ev.lastEventId) {
        state.lastEventId = ev.lastEventId;
        localStorage.setItem('lastEventId', state.lastEventId);
      }
      try{ await fn(ev); } catch(e){ console.warn(e); }
    });
  }

  es.onerror = ()=>{
    // auto-reconnect by browser; surface only if persistent
  };
}

// UI wiring
el('btnLogin').onclick = async ()=>{
  try{
    const username = el('loginUser').value.trim();
    const password = el('loginPass').value;
    const res = await api('/api/login', {method:'POST', body:{username, password}});
    state.user = res.user;
    state.csrf = res.user.csrf;
    state.settings = res.user.settings;
    applyBrand(state.settings);
    el('brandOrg').textContent = `${state.user.display_name} · ${state.user.role}`;
    setLoggedInUI(true);
    await loadIncidents();
    buildNav();
    connectSSE();
    render();
  } catch(e){ toast('Login fehlgeschlagen', e.message); }
};

el('btnLogout').onclick = async ()=>{
  try{ await api('/api/logout', {method:'POST'}); } catch(e){}
  state.user=null; state.csrf=null; state.incidentId=null;
  if (state.sse) state.sse.close();
  setLoggedInUI(false);
  toast('Logout','OK');
};

el('btnRefresh').onclick = refreshAll;
el('incidentSelect').onchange = async ()=>{
  state.incidentId = el('incidentSelect').value || null;
  localStorage.setItem('incidentId', state.incidentId||'');
  await render();
};

boot();
